//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.Controls 1.0

DialogItem {
    id : message

    property bool rejectOnClickOutside : false
    property list<DialogButton> buttons : [
        DialogButton {
            text : qsTrId("c4260415c3a56cd3")
            onClicked : {
                p_.indexFocusedButton = -1;
                message.accept(true);
            }
        },
        DialogButton {
            text : qsTrId("37ae1dd525a0ce7a")
            onClicked : {
                p_.indexFocusedButton = -1;
                message.reject();
            }
        }
    ]
    property int       primaryButtonIndex   : 0
    property string    title                : ""
    property string    instructionText      : ""
    property int       contentWidth         : 400
    property int       type

    QtObject {
        id : p_

        property int indexFocusedButton : -1

        onIndexFocusedButtonChanged : {
            if (indexFocusedButton === -1 ||
                indexFocusedButton > repeater.model.length - 1)
                return;

            repeater.itemAt(p_.indexFocusedButton).focus = true;
        }
    }

    anchors.fill : parent ? parent : undefined

    focus : true

    Keys.onEnterPressed : {
        repeater.itemAt(primaryButtonIndex).clicked();
        event.accepted = true;
        p_.indexFocusedButton = -1;
    }
    Keys.onReturnPressed : {
        repeater.itemAt(primaryButtonIndex).clicked();
        event.accepted = true;
        p_.indexFocusedButton = -1;
    }
    Keys.onEscapePressed : {
        message.reject();
        event.accepted = true;
        p_.indexFocusedButton = -1;
    }
    Keys.onLeftPressed : {
        p_.indexFocusedButton--;

        if (p_.indexFocusedButton < 0)
            p_.indexFocusedButton = repeater.model.length - 1;
    }
    Keys.onRightPressed : {
        p_.indexFocusedButton++;

        if (p_.indexFocusedButton > repeater.model.length - 1)
            p_.indexFocusedButton = 0;
    }

    onEnabledChanged : {
        if (enabled) {
            // reset repeater Buttons model to get proper states on open
           repeater.model = message.buttons;
        }
    }

    Rectangle {
        id : cover
        color : "#11000000"
        anchors.fill : parent
        opacity : message.open ? 1 : 0

        Behavior on opacity {
            NumberAnimation {
                duration : 200
            }
        }

        MouseArea {
            anchors.fill : parent
            acceptedButtons : Qt.LeftButton | Qt.RightButton
            hoverEnabled : true
            onWheel : {
                // catch wheel
            }
            onClicked : {
                if (rejectOnClickOutside) {
                    message.reject();
                    p_.indexFocusedButton = -1;
                }
            }
        }
    }

    BorderImage {
        id : shadow
        // 9-slice-margins : 18px
        // shadow offset : 13px
        // add 2*offset + 2*(margins-(margins-offset)) to width
        width : content.width + 26 + 32
        height : content.height + 26 + 32
        anchors.centerIn : parent
        opacity : message.open ? 1 : 0
        border {
            left : 18 + 13
            top : 18 + 13
            right : 18 + 13
            bottom : 18 + 13
        }
        horizontalTileMode: BorderImage.Stretch
        verticalTileMode: BorderImage.Stretch
        source: "qrc:/controls/images/dialog-shadow.png"
    }

    Rectangle {
        id : content

        border {
            width : 0
            color : UI.color(UI.SecondaryControlBorder)
        }
        anchors.centerIn : parent
        width : contentWidth
        height : buttonBar.height + header.height
        opacity : message.open ? 1 : 0

        Behavior on opacity {
            NumberAnimation {
                duration : 200
            }
        }

        Behavior on height {
            NumberAnimation {
                target: content
                property: "height"
                duration: 200
                easing.type: Easing.InOutQuad
            }
        }

        MouseArea {
            anchors.fill : parent
            acceptedButtons : Qt.LeftButton | Qt.RightButton
            z : -1
        }

        Rectangle {
            id : header

            property int contentHeight : {
                var height = 14 + title.height;

                if (instructionLabel.visible) {
                    height += 26 + instructionLabel.height;
                } else {
                    height += 13;
                }

                return height;
            }


            color : UI.color(UI.DialogHeader)
            anchors {
                top : parent.top
                left : parent.left
                right : parent.right
                margins : 0
            }
            height : contentHeight

            Image {
                id : typeIcon
                visible : message.type !== DialogItem.TypeNormal
                width : 36
                height : 36
                source : {
                    if (message.type === DialogItem.TypeWarning)
                        return "qrc:/controls/icons/warning.png"

                    if (message.type === DialogItem.TypeError)
                        return "qrc:/controls/icons/error.png"

                    if (message.type === DialogItem.TypeOk)
                        return "qrc:/controls/icons/status-ok.png"

                    return "";
                }
                x : 13
                y : 12
            }

            Text {
                id : title
                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : 18
                }
                color : UI.color(UI.DialogHeaderText)
                x : typeIcon.visible ? 15 + typeIcon.width : 15
                y : 14
                text : message.title
                width : parent.width - 2*instructionLabel.x
                wrapMode : Text.WordWrap
            }
            Text {
                id : instructionLabel

                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : 13
                }
                color : UI.color(UI.DialogHeaderText)
                x : 15
                y : title.y + title.height + 13
                text : message.instructionText
                width : parent.width - 18
                wrapMode : Text.WordWrap
                visible : message.instructionText.length > 0
            }
        }

        Item {
            id : buttonBar

            anchors {
                bottom : parent.bottom
                left : parent.left
                leftMargin : 7
                right : parent.right
                rightMargin : 7
            }
            height : 48

            Row {
                anchors {
                    right : parent.right
                    rightMargin : 8
                    verticalCenter : parent.verticalCenter
                }
                spacing : 4

                Repeater {
                    id : repeater
                    delegate : PushButton {
                        text : modelData.text
                        enabled : modelData.enabled
                        isPrimary : index === message.primaryButtonIndex
                        onClicked : modelData.clicked()
                        visible : modelData.visible
                    }
                }
            }
        }
    } // content
}
